pragma Singleton

import QtQml 2.12
import Lomiri.SystemSettings.Update 1.0
import "i18nd.js" as I18nd

QtObject {
    id: root

    property QtObject d: QtObject {
        property var channelMapCache: null
        property var pendingPromiseResolveFuncs: []

        property Connections conn: Connections {
            target: UpdateManager
            onStatusChanged: {
                switch (UpdateManager.status) {
                case UpdateManager.StatusCheckingImageUpdates:
                case UpdateManager.StatusCheckingAllUpdates:
                    break;
                default:
                    d.updateChannelMap();
                    break;
                }
            }
        }

        function updateChannelMap() {
            let result = {};
            let channels = SystemImage.getChannels();

            for (let chName of channels) {
                let parsed = parseChannel(chName);
                parsed.hidden = false; // TODO
                result[chName] = parsed;
            }

            d.channelMapCache = result;
            for (let resolv of d.pendingPromiseResolveFuncs)
                resolv(result);
            d.pendingPromiseResolveFuncs = [];
        }
    }

    function parseChannel(channel) {
        /*
         * FIXME: this code relies on channels being named in a certain way.
         * Suported format:
         *  - 20.04/arm64/android9plus/devel
         *  - ubports-touch/16.04/rc
         */

        /* e.g. 16.04, 20.04, 24.04-1.x */
        const seiresRegexp = /^[0-9]{2}\.[0-9]{2}(-[0-9a-z.]+)?$/;
        const prettyStabilities = {
            "stable": I18nd.tr("Stable"),
            "rc": I18nd.tr("Release candidate"),
            "devel": I18nd.tr("Development"),
            "daily": I18nd.tr("Daily"),
        };

        let c = channel.split('/');
        let series, prettySeries;
        if (seiresRegexp.test(c[0]))
            series = prettySeries = c[0];
        else if (c[0].startsWith("ut")) {
            series = c[0].substring(2);

            /* Special case for UTNext */
            if (series = "next") {
                series = "99.99";
                prettySeries = "Next";
            }
        } else if (c[1] && seiresRegexp.test(c[1]))
            series = prettySeries = c[1];
        else {
            series = '16.04';
            prettySeries = I18nd.tr("Unknown series");
        }

        let stability = c[c.length - 1];

        let prettyStability = stability;
        if (prettyStabilities[stability])
            prettyStability = prettyStabilities[stability];

        return {
            series,
            stability,
            prettyName: `${prettySeries} ${prettyStability}`,
        };
    }

    function getAvailableChannels() {
        if (d.channelMapCache)
            return Promise.resolve(d.channelMapCache);

        return new Promise(function (resolv, reject) {
            switch (UpdateManager.status) {
            case UpdateManager.StatusCheckingImageUpdates:
            case UpdateManager.StatusCheckingAllUpdates:
                // Don't queue another check if one is already in progress.
                break;
            default:
                UpdateManager.check(UpdateManager.CheckImage);
                break;
            }

            d.pendingPromiseResolveFuncs.push(resolv);
        });
    }

    function switchChannel(channel) {
        return new Promise(function (resolv, reject) {
            // Cancel all ongoing updates before switching
            SystemImage.cancelUpdate()

            SystemImage.setSwitchChannel(channel);

            if (channel === SystemImage.channelName) {
                SystemImage.setSwitchBuild(SystemImage.currentBuildNumber);
            } else {
                SystemImage.setSwitchBuild(0);
            }

            UpdateManager.check(UpdateManager.CheckImage);
            d.pendingPromiseResolveFuncs.push(function(chInfoMap) {
                // Discard chInfoMap for switchChannel() result.
                resolv();
            })
        });
    }
}
