/*
 * This file is part of lomiri-system-settings-system-update
 *
 * Copyright (C) 2025 UBports Foundation
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

import QtQml 2.12

import GSettings 1.0

import Lomiri.SystemSettings.Update 1.0

QtObject {
    id: releaseUpgradeManager

    property var currentVersion: null
    property var availableUpgrade: null
    // `null` = loading, "" = not loading.
    property var releaseHighlight: ""

    readonly property QtObject d: QtObject {
        function simpleFetch(url) {
            return new Promise(function (resolve, reject) {
                let req = new XMLHttpRequest();
                req.onreadystatechange = function () {
                    if (req.readyState != XMLHttpRequest.DONE)
                        return;

                    const status = req.status;
                    if (status === 0 || (status >= 200 && status < 400)) {
                        resolve(req.responseText);
                    } else {
                        let error =
                            new Error(`Fetching ${url} returns ${req.status} ` +
                                      `(${req.statusText})`);
                        error.status = req.status;
                        error.statusText = req.statusText;

                        reject(error);
                    }
                };

                req.open('GET', url);
                req.send();
            });
        }

        // TODO: this really should be a system-level settings, but this works
        // for now with minimal complexity.
        readonly property GSettings settings: GSettings {
            schema.id: "io.ubuntu-touch.system-settings.update"
        }

        property var lastSuccessfulCheck: new Date(0)
        function shouldRecheck() {
            // Only check 30 minutes after last successful check.
            const CHECK_INTERVAL_MS = 30 * 60 * 1000;
            
            let now = new Date();
            return now - lastSuccessfulCheck > CHECK_INTERVAL_MS;
        }

        readonly property string releaseHighlightUrl:
            availableUpgrade && availableUpgrade.releaseHighlightUrl
                ? availableUpgrade.releaseHighlightUrl
                : ''

        onReleaseHighlightUrlChanged: {
            if (!releaseHighlightUrl) {
                releaseUpgradeManager.releaseHighlight = "";
                return;
            }

            releaseUpgradeManager.releaseHighlight = null;
            simpleFetch(releaseHighlightUrl).then(function(response) {
                releaseUpgradeManager.releaseHighlight = response;
            }).catch(function(error) {
                console.warn(error);
                releaseUpgradeManager.releaseHighlight = "";
            });
        }

        property string newChannel: ''
    }

    function check(force) {
        if (!force && !d.shouldRecheck())
            return;

        releaseUpgradeManager.availableUpgrade = null;
        Promise.all([
            d.simpleFetch(d.settings.metareleaseUri),
            ChUtils.getAvailableChannels(),
        ]).then(function([response, siChannels]) {
            // TODO: validation?
            let metarelease = JSON.parse(response);
            let currentChannel = SystemImage.getSwitchChannel();
            let availableUpgrade = null;

            // Search from back to front because the list is sorted from oldest
            // to newest.
            for (let i = metarelease.length - 1; i >= 0; i--) {
                if (currentChannel in metarelease[i].systemImageChannels) {
                    releaseUpgradeManager.currentVersion = metarelease[i];
                    if (i + 1 < metarelease.length) {
                        availableUpgrade = metarelease[i + 1]
                    }
                    break;
                }
            }

            if (!availableUpgrade)
                return;

            if (availableUpgrade.supportStatus == "development" &&
                !d.settings.offerDevelopmentRelease)
            {
                // Since this is pretty common (and `console.debug()` prints by
                // default), don't log it.
                return;
            }

            if (availableUpgrade.upgradeBroken) {
                console.warn(`Upgrade to ${availableUpgrade.version} is ` +
                             `available, but is specified as broken ` +
                             `("${availableUpgrade.upgradeBroken}")`);
                return;
            }

            let currentChannelInfo =
                currentVersion.systemImageChannels[currentChannel];
            let newChannel = null;

            for (const [channel, info]
                 of Object.entries(availableUpgrade.systemImageChannels))
            {
                if (currentChannelInfo.stability === info.stability &&
                    currentChannelInfo.variant === info.variant)
                {
                    newChannel = channel;
                    break;
                }
            }

            if (!newChannel) {
                console.warn(`Can't find corresponding channel of ` +
                             `${currentChannel} in ${availableUpgrade.series}`);
                return;
            }

            if (!(newChannel in siChannels)) {
                console.warn(`Channel ${newChannel} is not available to ` +
                             `switch to.`);
                return;
            }

            releaseUpgradeManager.availableUpgrade = availableUpgrade;
            d.newChannel = newChannel;
            d.lastSuccessfulCheck = new Date();
        }).catch(function(error) {
            // TODO: there's not really a path way for error...
            console.warn(error);
        });
    }

    function startUpgrade() {
        if (!availableUpgrade) {
            return Promise.reject(
                new Error("startUpgrade() called without availableUpgrade"));
        }

        return ChUtils.switchChannel(d.newChannel);
    }
}
