/*
 * Copyright (C) 2025 UBports Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtQuick.Layouts 1.12
import Lomiri.Components 1.3
import Lomiri.Components.Popups 1.3
import Lomiri.Content 1.3
import QtMultimedia 5.6
import Lomiri.Contacts 0.1

Page {
    id: root
    objectName: "contactSoundsPage"

    property var activeTransfer
    property string currentSoundFile: ""
    property int currentRingtoneType: Ringtone.AudioRingtoneUrl

    property var systemSoundList : RingtoneModel.listSounds(RingtoneModel.systemRingtonePaths)
    property var customSoundList: RingtoneModel.listSounds([RingtoneModel.customRingtonePath])

    function selectModel() {
        let soundIdx = systemSoundList.indexOf(root.currentSoundFile);
        if (soundIdx > -1) {
            section.selectedIndex = 0
        } else {
            // look at the custom sound list, if found select the section
            soundIdx = customSoundList.indexOf(root.currentSoundFile);
            if (soundIdx > -1) {
                section.selectedIndex = 1
            }
        }
        soundList.currentIndex = soundIdx
        delayScrollTimer.start()
    }


    onCurrentSoundFileChanged: soundSelected(currentSoundFile)

    signal soundSelected(string file)
    signal closeRequested()

    function requestFileDelete(path) {
        var dialog = PopupUtils.open(deleteConfirm, root)
        dialog.canceled.connect(function() {
            PopupUtils.close(dialog)
        });
        dialog.accepted.connect(function() {
            RingToneModel.removeFile(path)
            root.currentSoundFile = ""
            PopupUtils.close(dialog)
        });
    }

    function onImportedFile(uri) {
        const newFile = uri.toString().replace("file:///", "/")

        root.currentSoundFile = newFile
        console.log("Set custom Sound URL: " + root.currentSoundFile)
        // reload custom list sound
        root.customSoundList = RingtoneModel.listSounds([RingtoneModel.customRingtonePath])
    }

    header: PageHeader {
        title: i18n.dtr("lomiri-addressbook-app", "Select a custom sound")
        leadingActionBar {
            actions: [
                Action {
                    text: i18n.tr("back")
                    iconName: "back"
                    onTriggered: {
                        soundEffect.stop()
                        root.closeRequested()
                    }
                }
            ]
        }

        trailingActionBar {
            actions: [
                Action {
                    text: i18n.tr("import")
                    iconName: "import"
                    visible: section.selectedIndex === 1
                    onTriggered: {
                        PopupUtils.open(pickerDialog, null)
                    }
                }
            ]
        }
        extension: Sections {
            id: section
            model: [ i18n.tr("System sounds"),  i18n.tr("Custom sounds") ]
        }
    }

    Timer {
        id: delayScrollTimer
        interval:200
        running: false
        repeat: false
        onTriggered: {
            // in case the auto positioning does not work for elements at the end of the list
            soundList.positionViewAtIndex(soundList.currentIndex, ListView.Center)
        }
    }

    Audio {
        id: soundEffect
        audioRole: MediaPlayer.NotificationRole

        function controlPlayback(fileURL) {
           if (source == fileURL && playbackState === Audio.PlayingState) {
               stop()
           } else {
               source = fileURL
               play()
           }
        }
    }

    Rectangle {
        anchors.fill: parent
        color: theme.palette.normal.background
    }

    Label {
        anchors.centerIn: parent
        text: i18n.tr("No sounds found")
        visible: soundList.count === 0
    }

    ColumnLayout {
        id: column
        anchors {
            top: header.bottom
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }

        ListView {
            id: soundList
            Layout.fillHeight: true
            Layout.fillWidth: true
            currentIndex: -1
            model: section.selectedIndex === 0 ? root.systemSoundList : root.customSoundList

            highlightMoveDuration: 1000
            flickDeceleration: 200
            highlightRangeMode: ListView.ApplyRange
            clip: true

            delegate: ListItem {
                id: soundItem
                height: layout.height
                divider.visible: false
                swipeEnabled: section.selectedIndex === 1

                property bool isChecked: modelData === currentSoundFile

                leadingActions: ListItemActions {
                    actions: [
                        Action {
                            iconName: "delete"
                            onTriggered: root.requestFileDelete(path)
                        }
                    ]
                }
                ListItemLayout {
                    id: layout
                    title.text: RingtoneModel.fileName(modelData)

                    Icon {
                        SlotsLayout.position: SlotsLayout.Trailing
                        width: units.gu(2)
                        height: width
                        name: "media-playback-pause"
                        visible: isChecked &&
                            soundEffect.playbackState === Audio.PlayingState
                        asynchronous: true
                    }

                    Icon {
                        id: tickIcon
                        SlotsLayout.position: SlotsLayout.Last
                        opacity: isChecked ? 1 : 0
                        width: units.gu(2)
                        height: width
                        name: "tick"
                        asynchronous: true
                    }
                }
                onClicked: {
                    root.currentSoundFile = modelData
                    soundEffect.controlPlayback("file://" + modelData)
                }
            }
        }
    }

    Component {
        id: pickerDialog

        Item {
            id: picker

            property var activeTransfer: null
            signal avatarReceived(string avatarUrl)
            signal destruction()

            anchors.fill: parent

            function hide() {
                picker.visible = false;
            }

            function show() {
                picker.visible = true;
            }

            ContentTransferHint {
                anchors.fill: parent
                activeTransfer: picker.activeTransfer
            }

            ContentPeerPicker {
                id: peerPicker

                anchors.fill: parent
                handler: ContentHandler.Source
                contentType: ContentType.Music
                showTitle: true

                onPeerSelected: {
                    peer.selectionType = ContentTransfer.Single
                    picker.activeTransfer = peer.request()
                }

                onCancelPressed: PopupUtils.close(picker);
            }

            Connections {
                id: contentHubConnection

                property var allowedExtensions :["wav","ogg","mp3"]

                target: picker.activeTransfer

                function onStateChanged() {
                    if (activeTransfer.state === ContentTransfer.Charged) {
                        if (activeTransfer.items.length > 0) {

                            var item = activeTransfer.items[0];
                            const isMusicFile = allowedExtensions.some((ext) => item.url.toString().endsWith(ext));
                            if (!isMusicFile) {
                                PopupUtils.open(unsupportedFile, root)
                                activeTransfer.finalize()
                                return;
                            }

                            var toneUri;
                            if (item.move(RingtoneModel.customRingtonePath)) {
                                toneUri = item.url;
                            } else {
                                PopupUtils.open(importError, root)
                                activeTransfer.finalize()
                                return;
                            }

                            console.log("moved file to:", toneUri)
                            PopupUtils.close(picker)

                            root.onImportedFile(toneUri);
                        }
                        activeTransfer.finalize()
                    }
                }
            }
        }
    }

    Component {
        id: deleteConfirm

        Dialog {
            id: dialog

            property int threadCount: 0

            title: i18n.tr("Delete file")
            text: i18n.tr("Are you sure you want to delete this file ?")

            signal accepted()
            signal canceled()

            Column {
                anchors.left: parent.left
                anchors.right: parent.right
                spacing: units.gu(2)
                Row {
                    anchors.horizontalCenter: parent.horizontalCenter
                    spacing: units.gu(4)
                    Button {
                        text: i18n.tr("Cancel")
                        onClicked: dialog.canceled()
                    }
                    Button {
                        text: i18n.tr("Delete")
                        color: theme.palette.normal.negative
                        onClicked: dialog.accepted()
                    }
                }
            }
        }
    }

    Component {
        id: unsupportedFile
        Dialog {
            id: dialog

            title: i18n.tr("Unsupported file")

            text: i18n.tr("Sorry, only ogg, mp3 and wav file format is supported")

            Column {
                anchors.left: parent.left
                anchors.right: parent.right
                spacing: units.gu(2)
                Row {
                    anchors.horizontalCenter: parent.horizontalCenter
                    spacing: units.gu(4)
                    Button {
                        text: i18n.tr("ok")
                        onClicked: dialog.hide()
                    }
                }
            }
        }
    }

    Component {
        id: importError
        Dialog {
            id: dialog

            title: i18n.tr("Import error")

            text: i18n.tr("Sorry, an error occurred while importing")

            Column {
                anchors.left: parent.left
                anchors.right: parent.right
                spacing: units.gu(2)
                Row {
                    anchors.horizontalCenter: parent.horizontalCenter
                    spacing: units.gu(4)
                    Button {
                        text: i18n.tr("ok")
                        onClicked: dialog.hide()
                    }
                }
            }
        }
    }

    Component.onCompleted: {
        // select the correct section according to current sound
        selectModel()
    }

    Component.onDestruction: {
        if (soundEffect.playbackState === Audio.PlayingState) {
            soundEffect.stop()
        }
    }
}

