# Copyright 2015-2017 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import psutil
import shlex
import shutil
import subprocess

from .Libertine import BaseContainer
from . import utils
from libertine.ContainersConfig import ContainersConfig


def chown_recursive_dirs(path):
    uid = 0
    gid = 0

    if 'SUDO_UID' in os.environ:
        uid = int(os.environ['SUDO_UID'])
    if 'SUDO_GID' in os.environ:
        gid = int(os.environ['SUDO_GID'])

    if uid != 0 and gid != 0:
        for root, dirs, files in os.walk(path):
            for d in dirs:
                os.chown(os.path.join(root, d), uid, gid)
            for f in files:
                os.chown(os.path.join(root, f), uid, gid)

        os.chown(path, uid, gid)


class LibertineChroot(BaseContainer):
    """
    A concrete container type implemented using bubblewrap sandboxing
    tool - functionally similar to a plain old chroot running unprivileged.
    """

    def __init__(self, container_id, config, service):
        super().__init__(container_id, 'chroot', config, service)
        os.environ['FAKECHROOT_CMD_SUBST'] = '$FAKECHROOT_CMD_SUBST:/usr/bin/chfn=/bin/true'
        os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
        os.environ['FAKEROOTDONTTRYCHOWN'] = '1'
        os.environ['SYSTEMD_OFFLINE'] = '1'

        # Use fakeroot to run apt and dpkg commands
        self.apt_command_prefix = 'fakeroot ' + self.apt_command_prefix
        self.dpkg_command_prefix = 'fakeroot ' + self.dpkg_command_prefix

    def run_in_container(self, command_string):
        cmd_args = shlex.split(command_string)
        command_prefix = self._build_bwrap_command()

        args = shlex.split(command_prefix + ' ' + command_string)
        cmd = subprocess.Popen(args)
        return cmd.wait()

    def destroy_libertine_container(self, force):
        return self._delete_rootfs()

    def create_libertine_container(self, password=None, multiarch=False):
        # Save and temporary unset LD_PRELOAD to avoid missing libtls-padding.so spam
        ld_preload = os.environ.get('LD_PRELOAD', '')
        if 'libtls-padding.so' in ld_preload:
            del os.environ['LD_PRELOAD']

        # Use fakechroot to create base rootfs
        commands = [
            # Create the actual chroot
            "{} fakeroot debootstrap --verbose --variant=fakechroot {} {}".format(
                self._build_fakechroot_command(), self.installed_release, self.root_path),
            # Install fakeroot inside the container so it can run under bwrap
            "{} fakeroot chroot {} apt-get install -y fakeroot".format(
                self._build_fakechroot_command(), self.root_path),
        ]
        for command_line in commands:
            args = shlex.split(command_line)
            cmd = subprocess.Popen(args)
            cmd.wait()

            if cmd.returncode != 0:
                utils.get_logger().error(utils._("Failed to create container"))
                self.destroy_libertine_container()
                return False

        # Remove symlinks as they are wrong outside of fakechroot
        utils.get_logger().info(utils._("Fixing chroot symlinks..."))
        os.remove(os.path.join(self.root_path, 'dev'))
        os.remove(os.path.join(self.root_path, 'proc'))

        # Restore real ldconfig and ldd
        ldconfig_path = os.path.join(self.root_path, 'usr/sbin/ldconfig')
        os.rename(ldconfig_path + '.REAL', ldconfig_path)
        ldd_path = os.path.join(self.root_path, 'usr/bin/ldd')
        os.rename(ldd_path + '.REAL', ldd_path)

        # Make sure user-data dir exists before calling run_in_container
        self._create_libertine_user_data_dir()

        # Fix dangling symlinks due to fakechroot
        # See https://github.com/dex4er/fakechroot/issues/45
        self.run_in_container('update-alternatives --all --skip-auto')

        # Add universe, multiverse, and -updates to the chroot's sources.list
        if (self.architecture == 'armhf' or self.architecture == 'arm64'):
            archive = "deb http://ports.ubuntu.com/ubuntu-ports "
        else:
            archive = "deb http://archive.ubuntu.com/ubuntu "

        utils.get_logger().info(utils._("Updating chroot's sources.list entries..."))

        with open(os.path.join(self.root_path, 'etc', 'apt', 'sources.list'), 'a') as fd:
            fd.write(archive + self.installed_release + "-updates main\n")
            fd.write(archive + self.installed_release + " universe\n")
            fd.write(archive + self.installed_release + "-updates universe\n")
            fd.write(archive + self.installed_release + " multiverse\n")
            fd.write(archive + self.installed_release + "-updates multiverse\n")

        self.update_locale()

        if multiarch and self.architecture == 'amd64':
            utils.get_logger().info(utils._("Adding i386 multiarch support..."))
            self.run_in_container(self.dpkg_command_prefix + " --add-architecture i386")

        utils.get_logger().info(utils._("Updating the contents of the container after creation..."))
        self.update_packages()

        for package in self.default_packages:
            if not self.install_package(package, update_cache=False):
                utils.get_logger().error(utils._("Failure installing '{package_name}' during container creation".format(package_name=package)))
                self.destroy_libertine_container()
                return False

        # We need to add the UBports HTTPS repo after apt-transport-https is
        # installed, so that update will not fail on not supporting HTTPS.
        self.run_in_container('fakeroot apt-get install -y gnupg2 gpgv wget')
        self.run_in_container('wget https://repo.ubports.com/keyring.gpg -O /tmp/ubports.key')
        self.run_in_container('fakeroot apt-key add /tmp/ubports.key')
        with open(os.path.join(self.root_path, 'etc', 'apt', 'sources.list.d', 'ubports.list'), 'w+') as fd:
            fd.write('\n\n# UBports repo to match rootfs\n')
            fd.write('deb https://repo2.ubports.com {} main\n'.format(self.installed_release))
        self.update_packages()

        # Install libtls-padding.so and restore LD_PRELOAD if needed
        if 'libtls-padding.so' in ld_preload:
            self.run_in_container('fakeroot apt-get install -y libtls-padding0')
            os.environ['LD_PRELOAD'] = ld_preload

        # Check if the container was created as root and chown the user directories as necessary
        chown_recursive_dirs(utils.get_libertine_container_home_dir(self.container_id))

        super().create_libertine_container()

        return True

    def _build_fakechroot_command(self):
        cmd = 'fakechroot'

        if utils.is_snap_environment():
            cmd = "{} -b {}/usr/sbin".format(cmd, os.environ['SNAP'])

        return cmd

    def _build_bwrap_command(self):
        bwrap_cmd = shutil.which('bwrap')
        if not bwrap_cmd:
            raise RuntimeError(utils._('executable bwrap not found'))

        bwrap_cmd += ' --bind "{}" /'.format(self.root_path)
        bwrap_cmd += ' --dev-bind /dev /dev --proc /proc --ro-bind /sys /sys'
        bwrap_cmd += ' --bind /run /run --bind /tmp /tmp'

        # Bind-mount extrausers on the phone
        if os.path.exists("/var/lib/extrausers"):
            bwrap_cmd += " --ro-bind /var/lib/extrausers /var/lib/extrausers"

        home_path = os.environ['HOME']

        # Use a separate user home, but bind-mount common XDG directories inside
        bind_mounts = ' --bind "{}" "{}"'.format(
            utils.get_libertine_container_home_dir(self.container_id), home_path)

        mounts = self._sanitize_bind_mounts(utils.get_common_xdg_user_directories() + \
                                            ContainersConfig().get_container_bind_mounts(self.container_id))
        for user_dir in utils.generate_binding_directories(mounts, home_path):
            if os.path.isabs(user_dir[1]):
                path = user_dir[1]
            else:
                path = os.path.join(home_path, user_dir[1])

            bind_mounts += ' --bind "{}" "{}"'.format(user_dir[0], path)

        bwrap_cmd += bind_mounts

        # Bind-mount XDG_RUNTIME_DIR for mirclient/Wayland connection
        xdg_runtime_dir = os.environ['XDG_RUNTIME_DIR']
        bwrap_cmd += ' --bind "{0}" "{0}"'.format(xdg_runtime_dir)

        user_dconf_path = os.path.join(home_path, '.config', 'dconf')
        if os.path.exists(user_dconf_path):
            bwrap_cmd += " --bind {0} {0}".format(user_dconf_path)

        return bwrap_cmd

    def _sanitize_bind_mounts(self, mounts):
        return [mount.replace(" ", "\\ ").replace("'", "\\'").replace('"', '\\"') for mount in mounts]

    def start_application(self, app_exec_line, environ):
        # Workaround issue where a custom dconf profile is on the machine
        if 'DCONF_PROFILE' in environ:
            del environ['DCONF_PROFILE']

        bwrap_cmd = self._build_bwrap_command()

        args = shlex.split(bwrap_cmd)
        args.extend(app_exec_line)
        return psutil.Popen(args, env=environ)

    def finish_application(self, app):
        app.wait()
