#!/usr/bin/python3

# Copyright (c) 2025 UBports Foundation
# SPDX-License-Identifier: GPL-2-or-later

import argparse
import sys

from gi.repository import GLib, Gio


class App:
    args: argparse.Namespace
    mainloop: GLib.MainLoop
    unit_object_proxy: Gio.DBusProxy

    def parse_arguments(self):
        parser = argparse.ArgumentParser(
            description="Wait for activation of a Systemd unit efficiently"
        )

        parser.add_argument(
            "--system", action="store_true", help="Connect to system manager (default)"
        )

        parser.add_argument(
            "--user", action="store_true", help="Connect to user service manager"
        )

        parser.add_argument(
            "--failed-state-is-success",
            action="store_true",
            help="If the unit fails, don't return failure from command",
        )

        parser.add_argument("unit", help="Systemd unit to watch")

        args = parser.parse_args()

        if args.system and args.user:
            parser.error("'--system' and '--user' can't be specified together.")

        return args

    def on_g_properties_changed(
        self, proxy, changed_properties, invalidated_properties
    ):
        for changed_prop_name in changed_properties.unpack():
            if changed_prop_name == "ActiveState":
                self.on_activestate_changed()
                break

    def on_activestate_changed(self):
        active_state = self.unit_object_proxy.get_cached_property(
            "ActiveState"
        ).unpack()
        match active_state:
            case "active":
                print(active_state)
                sys.exit(0)
            case "inactive" | "failed":
                print(active_state)

                if self.args.failed_state_is_success:
                    sys.exit(0)
                else:
                    sys.exit(1)
        # Do nothing for "activating", "deactivating", "maintenance",
        # "reloading", "refreshing".

    def __init__(self):
        self.args = self.parse_arguments()
        self.mainloop = GLib.MainLoop.new(None, False)

        bus = Gio.bus_get_sync(
            Gio.BusType.SESSION if self.args.user else Gio.BusType.SYSTEM,
            None,  # Cancellable
        )

        # Systemd requires that we call Subcribe() for signals to be emitted.
        # We don't need to call Unsubscribe() as Systemd does track us exiting.
        bus.call_sync(
            "org.freedesktop.systemd1",
            "/org/freedesktop/systemd1",
            "org.freedesktop.systemd1.Manager",
            "Subscribe",
            None,  # parameters
            None,  # reply_type
            Gio.DBusCallFlags.NONE,
            -1,  # timeout_msec
            None,  # cancellable
        )

        unit_object_path = bus.call_sync(
            "org.freedesktop.systemd1",
            "/org/freedesktop/systemd1",
            "org.freedesktop.systemd1.Manager",
            "GetUnit",
            GLib.Variant("(s)", (self.args.unit,)),  # parameters
            GLib.VariantType("(o)"),  # reply_type
            Gio.DBusCallFlags.NONE,
            -1,  # timeout_msec
            None,  # cancellable
        ).unpack()[0]

        self.unit_object_proxy = Gio.DBusProxy.new_sync(
            bus,
            # PropertyChanged signal is not covered by this flag.
            Gio.DBusProxyFlags.DO_NOT_CONNECT_SIGNALS
            | Gio.DBusProxyFlags.GET_INVALIDATED_PROPERTIES,
            None,  # GDBusInterfaceInfo
            "org.freedesktop.systemd1",
            unit_object_path,
            "org.freedesktop.systemd1.Unit",
            None,  # cancellable
        )

        self.unit_object_proxy.connect(
            "g-properties-changed", self.on_g_properties_changed
        )

        GLib.idle_add(self.on_activestate_changed)

    def run(self):
        self.mainloop.run()


if __name__ == "__main__":
    app = App()
    app.run()
