/*
 * Copyright (C) 2025 UBports Foundation
 *
 * This file is part of Lomiri System Settings
 *
 * Lomiri System Settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * Lomiri System Settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.12
import QtMultimedia 5.6
import SystemSettings 1.0
import Lomiri.Content 1.4
import Lomiri.Components 1.3
import Lomiri.SystemSettings.Sound 1.0

import "utilities.js" as Utilities

ItemPage {
    id: soundSelectorPage
    objectName: "soundSelectorPage"

    visible: false

    property string defaultSound: ""

    property string currentSound: ""

    property string oldSound: ""

    property var systemSoundPaths: []

    property string customSoundPath: ""

    property var systemSoundList : backendInfo.listSounds(systemSoundPaths)
    property var customSoundList: backendInfo.listSounds([customSoundPath])

    // Content-hub Import Properties
    property var activeTransfer: null

    signal soundSelected(string soundFile)

    function selectModel() {
        let soundIdx = systemSoundList.indexOf(soundSelectorPage.currentSound);
        if (soundIdx > -1) {
            console.log('found sound in systemSoundList at index', soundIdx)
            section.selectedIndex = 0
        } else {
            // look at the custom sound list, if found select the section
            soundIdx = customSoundList.indexOf(soundSelectorPage.currentSound);
            if (soundIdx > -1) {
                console.log('found sound in customSoundList at index', soundIdx)
                section.selectedIndex = 1
            }
        }
        _soundsRepeater.currentIndex = soundIdx
        delayScrollTimer.start()
    }

    header: PageHeader {
        id:header
        title: soundSelectorPage.title
        leadingActionBar.actions: [
            Action {
                iconName: "back"
                onTriggered: {
                    previewSoundPlayer.stop()
                    pageStack.removePages(soundSelectorPage)
                }
            }
        ]
        trailingActionBar.actions: [
            Action {
                id: saveAction
                objectName: "saveAction"
                iconName: "ok"
                enabled: soundSelectorPage.oldSound !== soundSelectorPage.currentSound && soundSelectorPage.currentSound.length > 0
                onTriggered: {
                    console.log('set new sound:' , soundSelectorPage.currentSound)
                    soundSelectorPage.soundSelected(soundSelectorPage.currentSound.replace("file://", ""));
                    pageStack.removePages(soundSelectorPage)
                }
            },
            Action {
                id: importAction
                objectName: "importAction"
                iconName: "add"
                visible: section.selectedIndex === 1
                onTriggered: {
                    pageStack.addPageToNextColumn(soundSelectorPage, chsoundSelectorPage, {soundPage: soundSelectorPage})
                }
            }
        ]
        extension: Sections {
            id: section
            model: [ i18n.tr("System sounds"),  i18n.tr("Custom sounds") ]
        }
    }

    LomiriSoundPanel { id: backendInfo }

    Timer {
        id: delayScrollTimer
        interval:200
        running: false
        repeat: false
        onTriggered: {
            // in case the auto positioning does not work for elements at the end of the list
            _soundsRepeater.positionViewAtIndex(_soundsRepeater.currentIndex, ListView.Center)
        }
    }

    ContentTransferHint {
        anchors.fill: parent
        activeTransfer: soundSelectorPage.activeTransfer
    }

    Connections {
        target: soundSelectorPage.activeTransfer
        function onStateChanged() {
            if (soundSelectorPage.activeTransfer.state === ContentTransfer.Charged) {
                var importItem = soundSelectorPage.activeTransfer.items[0]
                if (importItem && importItem.copy(customSoundPath)) {

                    soundSelectorPage.currentSound = importItem.url
                    console.log("Set custom Sound URL: " + importItem.url)
                    // reload custom list sound
                    soundSelectorPage.customSoundList = backendInfo.listSounds([customSoundPath])
                } else {
                    console.warn('failed to store file',  importItem.url, 'to', customSoundPath)
                }
                soundSelectorPage.activeTransfer.finalize()
            }
        }
    }

    Connections {
        target: Qt.application
        function onActiveChanged() {
            if (Qt.application.state != Qt.ApplicationActive) {
                previewSoundPlayer.stop()
            }
        }
    }

    Component.onDestruction: {
        if (previewSoundPlayer) {
            previewSoundPlayer.stop()
        }
    }

    Audio {
        id: previewSoundPlayer
        audioRole: MediaPlayer.NotificationRole

        function controlPlayback(fileURL) {
            if (source == fileURL && playbackState === Audio.PlayingState) {
                stop()
            } else {
                source = fileURL
                play()
            }
        }
    }

    Item {
        id: content

        anchors {
            top: header.bottom
            topMargin: units.gu(1)
            right: parent.right
            left: parent.left
            bottom: parent.bottom
        }

        Label {
            anchors.centerIn: parent
            visible: section.selectedIndex === 1 && _soundsRepeater.count === 0
            text: i18n.tr("Use the \"+\" button to import new sound")
        }

        ListView {
            id: _soundsRepeater

            anchors.fill: parent

            clip: true
            highlightMoveDuration: 1000
            highlightRangeMode: ListView.ApplyRange

            model: section.selectedIndex === 0 ? systemSoundList : customSoundList
            currentIndex: -1

            Component.onCompleted: {
                var scaleFactor = units.gridUnit / 8;
                maximumFlickVelocity = maximumFlickVelocity * scaleFactor;
                flickDeceleration = flickDeceleration * scaleFactor;
            }

            delegate: ListItem {
                id: _soundDelegate
                divider.visible: false
                width: _soundsRepeater.width - units.gu(2)
                objectName: "soundDelegate" + index

                property bool isChecked: modelData === currentSound

                height: _soundName.height + divider.height

                property var _leadingActions: ListItemActions {
                    actions: [
                        Action {
                            iconName: "delete"
                            onTriggered: {
                                previewSoundPlayer.stop()
                                if (!backendInfo.removeFile(modelData)) {
                                    console.warn('Failed to remove file:', modelData)
                                } else {

                                    if (oldSound === modelData) {
                                        // Reset settings, because settings is now invalid.
                                        console.log('reset selected sound to default sound', soundSelectorPage.defaultSound)
                                        soundSelectorPage.oldSound = soundSelectorPage.defaultSound
                                        soundSelectorPage.soundSelected(soundSelectorPage.defaultSound.replace("file://", ""));
                                    }

                                    if (_soundDelegate.isChecked) {
                                        // unset UI selection, because the UI selection is now invalid.
                                        soundSelectorPage.currentSound = "";
                                    }

                                    // reload custom list sound
                                    soundSelectorPage.customSoundList = backendInfo.listSounds([customSoundPath])
                                }
                            }
                        }
                    ]
                }

                // only allow to delete custom sound
                leadingActions: section.selectedIndex === 1 ? _leadingActions : null

                ListItemLayout {
                    id: _soundName
                    objectName: "soundName" + index

                    title.text: Utilities.buildDisplayName(modelData)

                    Icon {
                        SlotsLayout.position: SlotsLayout.Trailing
                        width: units.gu(2)
                        height: width
                        name: "media-playback-pause"
                        visible: isChecked &&
                                 previewSoundPlayer.playbackState === Audio.PlayingState
                        asynchronous: true
                    }

                    Icon {
                        id: tickIcon
                        SlotsLayout.position: SlotsLayout.Last
                        opacity: isChecked ? 1 : 0
                        width: units.gu(2)
                        height: width
                        name: "tick"
                        asynchronous: true
                    }
                }

                onClicked: {
                    soundSelectorPage.currentSound = modelData
                    previewSoundPlayer.controlPlayback("file://" + modelData)
                }
            }
        }
    }

    Component {
        id: chsoundSelectorPage
        Page {
            id: picker

            header: PageHeader {
                title: i18n.tr("Choose from")
            }

            property var soundPage

            ContentPeerPicker {
                id: peerPicker
                anchors.top: header.bottom
                anchors.topMargin: units.gu(1)
                handler: ContentHandler.Source
                contentType: ContentType.Music
                showTitle: false

                onPeerSelected: {
                    peer.selectionType = ContentTransfer.Single
                    soundPage.activeTransfer = peer.request()
                    pageStack.removePages(picker);
                }
                onCancelPressed: pageStack.removePages(picker);
            }
        }
    }

    Component.onCompleted: {
        // Record the current sound
        oldSound = currentSound
        // select the correct section according to current sound
        selectModel()
    }
}
